%% This script performs simulation studies in the paper

clear; close all; clc;
addpath(genpath(pwd))
rng('default')
rng(0)


%% Specify problem dimensions and output settings
% distribution family
family = 'Gaussian'; % 'uniform'; %'poisson'; %  'Gaussian';  %  'Cauchy'

simulationType = 'regression'; % 'location' %  'regression';
if strcmp(simulationType, 'location')
    % Table 1
    n = 100;
    p = 1;
    m = 10; %  choices in the paper: 10, 20, 30, 40
elseif strcmp(simulationType, 'regression')
    % Table 2
    n = 1000; 
    m = 1;
    p = 30; %  choices in the paper: 10, 20, 30, 40
else
    error('Wrong input')
end
lossfamily = 'Gaussian'; %'poisson'; or 'Gaussian';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Simulation data parameters
NumRuns = 50; % number of simulations (runs)
Output = 0;
BTrue = ones(p, m);
% parameters for outlier generation 
% addoutlier controls the percentage and outlier strength and leverage strength of outliers
addoutlier.Pct = 0;
addoutlier.outlierStrength = 0;
addoutlier.levStrength = 0;

% Specify a parameter value (B^0 in the paper, of size p x m) to calculate depth
if strcmp(simulationType, 'location')
    TargetType =  'equipoint-0.1'; %'equipoint-0'; % 'random'; % i.e., [0.1,0.1,...0.1] %;
elseif strcmp(simulationType, 'regression')
    TargetType = 'equipoint-0';  % 'random';
else
    error('Wrong input')
end

if strcmpi(TargetType, 'true')
    B_target = BTrue;
elseif strcmpi(TargetType, 'random')
    B_target = 0.1 * randn(p,m);  %(rand(1,m)-0.5); %1 * randn(p,m); (rand(1,m)-0.5);
elseif strcmpi(TargetType(1:9), 'equipoint')
    B_target = ones(p,m) * str2num(TargetType(11:end)); 
else
    error('Specify B')
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
tol = 1e-2;
tolgrad = 1e+0; 
maxiter = 5e+3;
problemsize = [n, p, m];

numOfInitValues = 10;   % This parameter controls the number of (random) initial values. 
                        % Multiple random starts, together with our optimization techniques,  
                        % are critical for tackling the nonconvexity.
                        % In some very challenging problems, one can consider using a larger numOfInitValues.

method = 'SAP:2ndAcc'; % method = 'SAP:apr1';   %  % The methods we have implemented: {'Manopt:tr', 'Manopt:cg', 'Manopt:bb', 'Manopt:rlbfgs', 'SAP:pr0', 'SAP:pr1', 'SAP:apr0', 'SAP:apr1', 'SAP:2ndAcc', 'SAP:rgg0', 'SAP:rgg1'};
optsVOpt = Funct_ParamsConfiguration(problemsize, tol, tolgrad, maxiter, numOfInitValues, [], method); 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Start the computation
disp(simulationType)
fprintf('NumRuns = %d, n = %d, m = %d, p = %d, family = %s. \n',  NumRuns, n, m, p, family);
fprintf('tolgrad = %.e, maxiter = %.e, numOfInitValues = %d \n', tolgrad, maxiter, numOfInitValues);
fprintf('================================================================================ \n');
fprintf('================================================================================ \n');

depthres = zeros(NumRuns, 1); % to compare different methods for comparisons in total
timeres = zeros(NumRuns, 1);
iternumes = zeros(NumRuns, 1);
% Generate multiple datasets, call each method, and summarize the results
for i = 1:NumRuns
    %% Generate some data
    fprintf('================================================================================ \n'); 
    fprintf('%dth simulation dataset starts ... \n', i);
    if strcmp(simulationType, 'location')
        X = ones(n,1);
        switch (family) % for simplicity, assume zero mean
            case 'Gaussian'      
                Y = randn(n,m);
            case 'uniform' % U(-3, 3)
                Y = (rand(n,m)-0.5) * 6 ;
            case 'Cauchy'  % standard Cauchy
                Y = trnd(1,n,m);               
        end
    elseif strcmp(simulationType, 'regression')
        X = [ones(n, 1), randn(n, p-1)];
        switch (family)
            case 'Gaussian'
                noise = randn(n, 1);
            case 'uniform' % U(-1, 1)
                noise = (rand(n, 1)- 0.5) * 2;
            case 'Cauchy' % standard Cauchy
                noise = trnd(1, n, 1);
        end
        B_true = ones(p, 1); 
        Y = X*B_true + noise;
    end
    % [X, Y] = Funct_geneGLMData(dataName, n, p, BTrue, addoutlier, family);
    Xt = X.'; 
    %--------------------------------------------------------------------------------%
    % Construct the R matrix
   
    R = Funct_GLMResidual_Gradient(Y, X, B_target, lossfamily); 
        % This is the only quantity that relies in the distribution or specified loss
    %--------------------------------------------------------------------------------%    
    [depth, time, Vopt, iternum] = Func_PHD01(X, R, Xt, [], optsVOpt); % Main function
    % fprintf('The depth using %s method is %.2f. Time elapsed %.3f. Iteration number is %d.\n', method, depth, time, iternum)
    depthres(i) = depth;
    timeres(i) = time;
    iternumes(i) = iternum;
end


fprintf('================================================================================ \n');
fprintf('The mean and median depths are %.3f and %.3f. \n', mean(depthres(:)), median(depthres(:)));
fprintf('The mean and median computational time are %.2f and %.2f. \n', mean(timeres(:)), median(timeres(:)));
fprintf('The mean and median iteration number are %.2f and %.2f. \n', mean(iternumes(:)), median(iternumes(:)));

