function [Bs_init] = aux_InitBs_SubsetGLM(X, Y, optsB, family, ZeroOut, seqBlock)
%==========================================================================
% This function generates some candidate Bs to be used in the nonconvex
% optimization algorithm in the GLM setup. Subset sampling is used in computation.
% We assume the first column of X is a vector of ones and so the associated
% row in B corresponds to the intercept(s).
%------------------------ Input Variables ---------------------------------
% X               - design matrix  
% Y               - The observation
% optsB           - Options for optimizing over B
% family          - GLM family, gaussian or poisson or binominal
% seqBlock        - sequential block sampling. Sequentially divide [1:n] into 
%                   seqBlock many blocks of roughly the same size. Within each block, 
%                   randomly sample subsetsize/seqBlock elments.
%------------------------ Output Variables --------------------------------
% Bs_init         - The initial of Bs 
%==========================================================================
% The number of generated Bs is given by 'optsB.numOfInitValues'.
if ~exist('seqBlock', 'var') || isempty(seqBlock)
    seqBlock = [];
end
if ~exist('ZeroOut', 'var')
    ZeroOut =[];
end
[n, p] = size(X);
m = size(Y,2);
if ~isfield(optsB, 'ridgeparam')
    ridgeparam = 1e-5; % 0; %
    % Becasue each estiamte is obtained from a subset of data which might
    % have collinearity, we add a mild ridge penalty to enhance numerical stability.
else
    ridgeparam = optsB.ridgeparam;
end

output = 0; % 1 % 0 %
if output == 0
    options.Display = 'off';
end

if p == 1 && all(X==1)
    locDepth = 1;
else
    locDepth = 0;
end

if locDepth
    subsetsize = 1; 
else
    subsetsize = min(10*p, n); % This parameter controls the size of each random subset; the value can be smaller
end

if locDepth % In the case of location depth, directly sample all initial values from the given matrix (Y)
    subsample_locDepth = randsample(n, optsB.numOfInitValues); % better than using a loop: (a) more efficient (b) without replacement
end

Bs_init = zeros(p, m, optsB.numOfInitValues - 2);

% construct a default function handle (associated with minfunc)
fHandle_DefaultFitting = @(x, y) minFunc( @(beta) Funct_GLMLoss_Gradient(beta, x, y, ridgeparam, family), zeros(size(x, 2), 1), options);

%%%%%
if strcmpi(family, 'poisson')
    Y(Y == 0) = 1e-5; % to alleviate instability when calling minfunc
end        
%%%%%

for j = 1:(optsB.numOfInitValues - 2)
    if locDepth % In the location depth case, we random sample rows of observations to generate candidate B's
        Bs_init( :, :,j) = Y(subsample_locDepth(j), :);
    else % GLM depth, multiple random starts
        %for each j, we obtain a random subset and then estimate B
%         subset = randsample(n, subsetsize); 
        subset = Funct_subsetSampling(n, subsetsize, seqBlock);

        % Model fitting on the restricted subset

        X_sub = X(subset, :);
        y_sub = Y(subset, :);
                
        Bs_init(:, :, j)  = Func_RestrictFit(fHandle_DefaultFitting, X_sub, y_sub, ZeroOut);
    end
end

if locDepth
    % We include the mean and median of all obs as initial values
    Bs_init = cat(3, mean(Y), median(Y), Bs_init); 
else
    % Add an estimate using GLM
    B_mean = Func_RestrictFit(fHandle_DefaultFitting, X, Y, ZeroOut);
    
    % Add an overall estimate using a somewhat robust method 
    % e.g.,  transform Y and then use quantile regression on (Y', X) to get a robust B
    if strcmpi(family, 'binomial') 
        SVMModel = @(x, y) fitcsvm(x, y);
        fHandle_AnotherFitting = @(SVMModel) SVMModel(x, y).Alpha' * SVMModel(x, y).SupportVectors;  
        B_median = Func_RestrictFit(fHandle_AnotherFitting, X , 2 * Y - 1, ZeroOut);
    else
        fHandle_AnotherFitting  = @(x, y) rq(x, y, 0.5);   % use quantile regression to fit model 
        B_median = Func_RestrictFit(fHandle_AnotherFitting, X, Funct_GLM_link(Y, family), ZeroOut);
    end
    Bs_init = cat(3, B_mean, B_median, Bs_init); 
end


end



function [subset] = Funct_subsetSampling(n, subsetsize, seqBlock)
if ~exist('seqBlock', 'var') || isempty(seqBlock) || seqBlock < 1
    subset = randsample(n, subsetsize);
else % In this case, perform sequential block sampling. 
     % That is, sequentially divide [1:n] into seqBlock many blocks of roughly the same size
     % Within each block, randomly sample subsetsize/seqBlock elments.
     % This way of sampling is useful for block diagonal designs. 
    setsize_withinBlock = floor(n / seqBlock);
    subsetsize_withinBlock = floor(subsetsize/seqBlock);
    subset = [];
    for blockInd = 1:seqBlock
        if blockInd < seqBlock
            persubsize = subsetsize_withinBlock; 
            persize = setsize_withinBlock;
        else
            persubsize = subsetsize - subsetsize_withinBlock * (seqBlock - 1);
            persize = n - setsize_withinBlock * (seqBlock - 1);
        end
        startInd = (blockInd-1)*persize + 1 ;
        subset = [subset; randsample(startInd : (startInd + persize - 1) , persubsize)]; 
    end
    if numel(subset) ~= subsetsize
        error('Wrong calculation')
    end
end
end


% function [est] = Func_RestrictFit(fHandle_Fitting, x, y, ZeroOut)
% % Fit a model resctricted to the nonzero entries of (vectorized) ZeroOut
%     [~, m] = size(y);
%     [n, p] = size(x);
%     oriX = x;
%     oriY = y;
%     if m > 1    
%         % We only consider a vector multvariate model with *indepdendence* between all responses
%         % and so the model fitting amounts to a single-response one on the vectorized model                
%         y = reshape(y, [n * m, 1]); % vectorize to single response variable
%         x = sparse(kron(eye(m), x));
%     end
%     if exist('ZeroOut', 'var') && ~isempty(ZeroOut)
%         est = zeros(p*m, 1);
%         nontrivialInds = find(ZeroOut(:) ~= 0); % use vectorized ZeroOut to get the restricted indices
%         x_restr = x(:, nontrivialInds);        
%         est(nontrivialInds) = fHandle_Fitting(x_restr, y);       
%     else
%         est = fHandle_Fitting(x, y);
%     end
%     if m > 1
%         est = reshape(est, [p, m]);
%     end
% end

