function [V, final_cost, info, i] = Func_APG(X, Xt, V_init, R, zeta, L, opts, ZeroOut)
%==========================================================================
% This function performs accelerated proximal gradient to solve the
% polished data depth problem.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% L               - Upper bound of the lipschitz constant
% initV           - Provide the starting value of unknown parameter V
% outs            - options for V optimization
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
%------------------------ Output Variables ---------------------------------
% V               - optimal solution of V
% final_cost      - function value at convergence
% info 	          - progress information
%==========================================================================
ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);
if ZeroOutEffective && ~isfield(ZeroOut, 'VMethod') 
    ZeroOut.VMethod = 'constrained'; % 'penalized' % 
end

if ZeroOutEffective
    V_init = V_init .* ZeroOut.mat;
    V_init = V_init/norm(V_init, 'fro');
end




f = zeros(1,opts.maxiter);
g = zeros(1,opts.maxiter);  %gradnorm
V_old = V_init;
Z = V_old;                     %Auxiliary variable in APG
t_cur = 1;
alpha = opts.alpha;
beta = opts.beta;


for i = 1:opts.maxiter
    t_new = (1 + sqrt(1+4*t_cur^2))/2;
    if   ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'penalized')
        [fz,gz] = Funct_ObjGradHessian_V(X, Xt, Z, R, zeta, ZeroOut);        
    else
        [fz,gz] = Funct_ObjGradHessian_V(X, Xt, Z, R, zeta); % We use the original object function
    end
       
    if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained') % compute the constrained V
        gz = gz .* ZeroOut.mat;
    end
    if strcmpi(opts.linesearch,'TRUE')
        rho = alpha*L;
        while true
            V_cur = Z - 1/rho*gz;
            if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained')% Remove the elements that are fixed to be 0 based on the constraint
                V_cur = V_cur .* ZeroOut.mat;
            end
            V_cur = V_cur/norm(V_cur,'fro');            
            [sur] = surrogate(V_cur, Z, fz, gz, rho);
            [f_new] = Funct_ObjGradHessian_V(X, Xt, V_cur, R, zeta);
            if f_new < sur
                break
            else
                if rho > L
                    break
                end
            end
            rho = rho*beta;            
        end
    else
        V_cur = Z - 1/L*gz;
        if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained')  
            V_cur = V_cur .* ZeroOut.mat;
        end
        V_cur = V_cur/norm(V_cur,'fro');
    end
    Z = V_cur + (t_cur-1)/t_new*(V_cur-V_old);
    [f_old,g_old] = Funct_ObjGradHessian_V(X,Xt,V_old,R,zeta);
    V_old = V_cur;
    t_cur = t_new;
    
    % Record progress and checking stopping criteria
    f(i) = f_old;
    g_old_proj = g_old - sum(sum(V_old.*g_old,2))*V_old;
    g(i) = norm(g_old_proj,'fro');
    if i > 1 && (g(i) < opts.tolgrad || abs(f(i) - f(i-1)) < opts.tol )
        f = f(1:i);
        g = g(1:i);
        break
    end
end
V = V_cur;
final_cost = f(end);
info.cost = f;
info.gradnorm = g;
end


%% Define the surrogate function
function [sur] = surrogate(V, V_cur, f_cur, g_cur, rho)
sur = f_cur +  sum(sum(g_cur.*(V-V_cur),2)) + rho/2*norm(V-V_cur,'fro')^2;%trace(g_cur.'*(V-V_cur))  + rho/2*norm(V-V_cur,'fro');
end
