function [V, final_cost, info, t] = Func_2ndAcc(X, Xt, V_init, R, zeta, L, opts, ZeroOut)
%==========================================================================
% This function performs the 2nd accelerated proximal gradient with line
% search. (The zeroout option can be used in depths with G restricting
% certain entries to zero.) 
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function  
% L               - Upper bound of the lipschitz constant
% V_init          - the starting value of unknown parameter V
% opts            - options for V optimization
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
%------------------------ Output Variables ---------------------------------
% V               - optimal solution of V
% final_cost      - function value at convergence
% info 	          - some information of the final iteratve (func val, gradient)
% t               - the number of iteration that cost
%==========================================================================
linesearch_criterion = 'MaxRScaled'; % 'lastR'; %'MaxR' %  

ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);
%if ZeroOut matrix is not exsist, it will never affect the algrithom
if ~isfield(opts, 'search_number')
    opts.search_number = 3;
end
if ZeroOutEffective && ~isfield(ZeroOut, 'VMethod') 
    ZeroOut.VMethod = 'constrained'; % 'penalized' 
end

[p, m] = size(V_init);
W_cur = zeros(p, m, opts.search_number);
V_cur = zeros(p, m, opts.search_number);

f = zeros(1,opts.maxiter);
g = zeros(1,opts.maxiter);  %gradnorm

if ZeroOutEffective
    V_init = V_init .* ZeroOut.mat;
    V_init = V_init/norm(V_init, 'fro');
end

V_old = V_init;
W_old = V_old;            
alpha = opts.alpha;
beta = opts.beta;
rho_min = L * alpha;    %rho_min = alpha;  %rho_old = L * alpha;

for t = 0 : opts.maxiter - 1
    lsCritvals = -1;  % line search criterion value
    rho_cur = rho_min;     %rho_cur=rho_old;

    for i = 1:opts.search_number 
        if i > 1
            rho_cur(i) = beta * rho_cur(i - 1);   
        end
        if t >= 1
            theta_cur(i) = (theta_old * sqrt(rho_old) * sqrt(rho_old * theta_old ^ 2 + 4 * rho_cur(i)) - rho_old * theta_old^2) / (2 * rho_cur(i)); %6
        else
            theta_cur(i) = 1;
        end
        U = (1 - theta_cur(i)) * V_old + theta_cur(i)* W_old;     

        if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'penalized')            
            [fz,gz] = Funct_ObjGradHessian_V(X, Xt, U, R, zeta, ZeroOut);
        else
            [fz,gz] = Funct_ObjGradHessian_V(X, Xt, U, R, zeta);  
        end

        if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained') 
            gz = gz .* ZeroOut.mat;
        end

        W_cur(:, :, i) = W_old - gz / (theta_cur(i) * rho_cur(i)); 

        if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained')
            W_cur(:, :, i) = W_cur(:, :, i) .* ZeroOut.mat;
        end
        W_cur(:, :, i) = W_cur(:, :, i) / norm(W_cur(:, :, i), 'fro');    

        V_cur(:, :, i) = (1 - theta_cur(i)) * V_old + theta_cur(i) * W_cur(:, :, i);  

        R_t = theta_cur(i) ^ 2 * rho_cur(i) * norm(W_cur(:, :, i) - W_old, 'fro') ^ 2 / 2 - Delta(V_cur(:, :, i), U, R, zeta, X, Xt, fz, gz) + (1 - theta_cur(i)) * Delta(V_old, U, R, zeta, X, Xt, fz, gz); 

        switch linesearch_criterion
            case 'MaxRScaled'
                lsCritvals(i) = R_t / (rho_cur(i) * theta_cur(i) ^ 2 );
            case 'MaxR'
                lsCritvals(i) = R_t;
            case 'lastR'
                lsCritvals(i) = R_t;
%                 if i == opts.search_number 
%                     lsCritvals(i) = Inf;
%                 end
                %lscrit = sort(lscrit, 'ascend'); % to make sure the largest number is the last element of the array

            otherwise
                error('not impliment yet')
        end

        if lsCritvals(i) >=0
            break
        end
    end
    if strcmp(linesearch_criterion, 'lastR')
        index = i;
    else
        [~,index] = max(lsCritvals);
    end
    rho_old = rho_cur(index);
    theta_old = theta_cur(index);
    W_old = W_cur(:, :, index);
    V_old = V_cur(:, :, index);

    % Check stopping criteria
    f(t+1) = fz;
    g_old_proj = gz - sum(sum(W_old.*gz,2))*W_old;
    g(t+1) = norm(g_old_proj,'fro');

    if t > 0 && ( g(t+1) < opts.tolgrad || abs(f(t+1) - f(t)) < opts.tol )    
    f = f(1:t+1);
    g = g(1:t+1);
    break
    end

end
V = V_old;
final_cost = f(end);
info.cost = f;
info.gradnorm = g;
t=t+1;
end

function[value] = Delta(V, U, R, zeta, X, Xt, fz, gz)
%==========================================================================
% 
% This function is used to culculate the Delta function in Algrithm 1-line
% 11
% 
%==========================================================================
[f1] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta);
diff = V - U;

cond1 = ~exist('fz', 'var') || isempty(fz);
cond2 = ~exist('gz', 'var') || isempty(gz);  
if cond1 && cond2
    [fz, gz] = Funct_ObjGradHessian_V(X, Xt, U, R, zeta);
elseif cond1 && ~cond2
    [fz] = Funct_ObjGradHessian_V(X, Xt, U, R, zeta);
elseif ~cond1 && cond2
    [fz, gz] = Funct_ObjGradHessian_V(X, Xt, U, R, zeta);
end

value = f1 - fz - sum(sum(gz.*diff, 2));
end


