function [V, grad_V] = aux_rrr_depth_BCD_V(X, Xt, init_V, R, P_perp, L, Q_perp, opts, zeta, rho_depth)
%==========================================================================
% This function performs the V-block optimization in calculating RRR depth.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% V_init          - initial value of V
% R               - residual matrix
% P_perp          - orthogonal complement to the P in compact SVD: B = P D Q'
% L               - the given value of L
% Q_perp          - orthogonal complement to the Q in compact SVD: B = P D Q'
% opts            - Options for optimizing V
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% rho_depth       - a scale parameter added in the objective function to
%                   improve numerical performance
%------------------------ Output Variables ---------------------------------
% V               - the optimal projection direction of residuals/influences 
% grad_V          - the gradient associated with the optimal V
%==========================================================================

if ~isfield(opts, 'V_method')
    opts.V_method = '2ndAcc';
end


switch opts.V_method
    case 'GD'
        opts.Lip_V = 1e-3; %2000000;
        [V, grad_V] = Funct_GD_V(X, Xt, init_V, R, P_perp, L, Q_perp, opts, zeta);
               
    case '2ndAcc'
        [n, p] = size(X);
        opts.tol = opts.V_tol;
        opts.alpha = 10^-(floor(log10(n/5)) + 3 + floor(log10(p))); %alpha.acc;
        opts.search_number = 5; % suffices
        opts.beta = 4;
        Lip = 2 * (norm(L, 'fro') ^ 2 / n + (1 / rho_depth)^2 * norm(X, 2) ^ 2 * norm(R, 2) ^ 2);
        opts.maxiter = opts.V_bcd_maxiter;
        
        funProj = @(X)Funct_Normalization_V(X);
        funObj = @(V)Funct_ObjGrad_V_RRR_Depth(X,Xt, V, R, P_perp, L, Q_perp, zeta, rho_depth);
        
        [V, ~, info, ~] = Func_2ndAcc_univ(init_V, Lip, funObj, funProj, opts); %(X, Xt, V_init, R, zeta, L, opts)
        grad_V = info.gradnorm;
end

V = V / norm(V, 'fro');

end 



function [V, grad_V] = Funct_GD_V(X, Xt, V, R, P_perp, L, Q_perp, opts, zeta)
    % A simple gradient decent for L optimization as a test
    n = size(X, 1);
    for i = 1 : opts.V_bcd_maxiter
        [~, grad_V] = Funct_ObjGrad_V_RRR_Depth(X, Xt, V, R, P_perp, L, Q_perp, zeta, rho_depth); %gradient with repect to V
        V = V - (1 / (2 * opts.Lip_V *  (norm(L, 'fro') ^ 2 / n + (1 / rho_depth)^2 * norm(X, 2) ^ 2 * norm(R, 2) ^ 2))) * grad_V;
        V = V / norm(V, 'fro');
    end
end

function [X_norm] = Funct_Normalization_V(X)
    X_norm = X / norm(X, 'fro');
end