function [L, grad_L] = aux_rrr_depth_BCD_L(X, Xt, V, R, P_perp, init_L, Q_perp, lambda, opts, zeta, rho_depth)
%==========================================================================
% This function performs the L-block optimization in calculating RRR depth.
% It is a constrained optimization problem.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% V               - the given value of V
% R               - residual matrix
% P_perp          - orthogonal complement to the P in compact SVD: B = P D Q'
% L_init          - initial value of L
% Q_perp          - orthogonal complement to the Q in compact SVD: B = P D Q'
% lambda          - the constraint parameter for || L ||_2 (constructed from B)
% opts            - Options for optimizing L
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% rho_depth       - a scale parameter added in the objective function to
%                   improve numerical performance
%------------------------ Output Variables ---------------------------------
% L               - the optimal (repametrized) slack variables/parameters 
% grad_L          - the gradient associated with the optimal L
%==========================================================================
if ~isfield(opts, 'L_method')
    opts.L_method = '2ndAcc';
end

switch opts.L_method
    case 'GD'
        opts.Lip_L = 200;
        [L, grad_L] = Funct_GD_L(X, V, R, P_perp, init_L, Q_perp, lambda, opts, zeta, rho_depth);
              
    case '2ndAcc'
        [n, p] = size(X);
        opts_L.tol = opts.L_tol;
        opts_L.tolgrad = opts.tolgrad;
        opts_L.alpha = 10^-(floor(log10(n/5)) + 2 + floor(log10(p))); %alpha.acc;
        opts_L.search_number = 3; % suffices
        opts_L.beta = 4;
        Lip = 1 / n; % 2 * (norm(L, 'fro') ^ 2 / n + norm(X, 2) ^ 2 * norm(R, 2) ^ 2);
        opts_L.maxiter = opts.L_bcd_maxiter;
        
        funProj = @(X)Funct_SingularProj_L(X, lambda);
        funObj = @(L)Funct_ObjGrad_L_RRR_Depth(X, V, R, P_perp, L, Q_perp, zeta, rho_depth);
        
        [L, ~, info, ~] = Func_2ndAcc_univ(init_L, Lip, funObj, funProj, opts_L); %(X, Xt, V_init, R, zeta, L, opts)
        grad_L = info.gradnorm;        



end

L = Funct_SingularProj_L(L, lambda);
end 



function [L, grad_L] = Funct_GD_L(X, V, R, P_perp, L, Q_perp, lambda, opts, zeta, rho_depth)
    % A simple gradient decent for L optimization (slow) as a test
    n = size(X, 1);

    for i = 1 : opts.L_bcd_maxiter
        [~, grad_L] = Funct_ObjGrad_L_RRR_Depth(X, V, R, P_perp, L, Q_perp, zeta, rho_depth); %gradient with repect to L
        L = L - (1 / (opts.Lip_L / n)) * grad_L;
        L = Funct_SingularProj_L(L, lambda);  %% Lambda function is requared
    end 
    
end


function [f, g] = Funct_ObjGrad_L_RRR_Depth(X, V, R, P_perp, L, Q_perp, zeta, rho_depth)
% This function calculates the gradient of the polished objective (with zeta) with respect to L
    [n, ~] = size(X);
    [objvec] = aux_Obj_RRR_Depth_objvec(X, V, R, P_perp, L, Q_perp, rho_depth);
    switch nargout
        case 1
            f1= aux_Phi(objvec, zeta);
            f = sum(f1);
        case 2
            [f1, g1] = aux_Phi(objvec, zeta);
            f = sum(f1);
            g = sum(g1) * (P_perp' * V * Q_perp) / n; %for L
    end
end

function [A_proj] = Funct_SingularProj_L(A, lambda)
% This function performs the SVD projection associated with the operator
% norm constraint by trucating large singular values.
[U, D, V] = svd(A);
%[~, dim2_D] = size(D);
if isvector(D) % dim2_D == 1
    D_diag = D(1);
else
    D_diag = diag(D);
end

D_diag(D_diag > lambda) = lambda; % threshold the diagonal elements by lambda
D(logical(eye(size(D)))) = D_diag; % change the corresponding matrix
A_proj = U * D * V';

end



