function [V, final_cost, info, i] = Func_PG(X, Xt, V_init, R, zeta, L, opts)
%==========================================================================
% This function performs proximal gradient with backtracking line search to solve the
% polished data depth problem.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% initV           - initial value of V
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% L               - lipschitz constant
% opts            - options for V optimization
%------------------------ Output Variables ---------------------------------
% V               - optimal solution of V
% final_cost      - function value at convergence
% info 	          - progress information
%==========================================================================
debug = 1;

f = zeros(1,opts.maxiter);
g = zeros(1,opts.maxiter);  %gradnorm
V = V_init;
alpha = opts.alpha;
beta = opts.beta;



for i = 1:opts.maxiter
    [f_cur,g_cur] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta);
    if strcmpi(opts.linesearch,'TRUE')
        rho = alpha*L;
        while true
            V_new = V - 1/rho*g_cur;
            V_new = V_new/norm(V_new,'fro');
            [sur] = surrogate(V_new, V, f_cur, g_cur, rho);
            [f_new] = Funct_ObjGradHessian_V(X, Xt, V_new, R, zeta);
            if f_new < sur
                V = V_new;
                break
            else
                if rho > L           %This is not needed, but it is always better to include a
                    V = V_new;       %termination in while loop due to possible numerical rounding
                    break            %error in computing f_new and sur.
                end
            end
            rho = rho*beta;
        end
    else
        V = V - (1/L)*g_cur;
        V = V/norm(V,'fro');
    end
    f(i) = f_cur;
    g_cur_proj = g_cur - sum(sum(V.*g_cur,2))*V; % the riemannian gradient
    % g_cur - trace(V.'*g_cur)*V;
    g(i) = norm(g_cur_proj,'fro');
    if i > 1 && g(i) < opts.tolgrad
        f = f(1:i);
        g = g(1:i);
        break
    end
    if i > 1 && abs(f(i)-f(i-1))<opts.tol
        f = f(1:i);
        g = g(1:i);
        break
    end
    if debug > 0 && i > 1 && f_cur > f(i-1) + 1e-12 %prevent numerical rounding error
        warning('Function value is not decreasing. Try a smaller step-size.')
    end    
end
final_cost = f(end);
info.cost = f;
info.gradnorm = g;
end


%% Define the surrogate function
function [sur] = surrogate(V, V_cur, f_cur, g_cur, rho)
sur = f_cur +  sum(sum(g_cur.*(V-V_cur),2)) + rho/2*norm(V-V_cur,'fro')^2;%trace(g_cur.'*(V-V_cur))  + rho/2*norm(V-V_cur,'fro');
end
