 function [depth, time, Vopt, iternum] = Func_PHD01(X, R, Xt, zetaGrid, opts, ZeroOut)
%==========================================================================
% This function computes the data depth of residuals associated with the 0-1 loss
% (Tukey depth) given the residuals computed from a target parameter by using the successive
% optimization (annealing), with the algorithm termed successive accelerated Procruestes rotation
% (SAP). 
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% R               - residual matrix
% zetaGrid        - the grid of the steepness for successive apporximations of the sign function
% opts            - Options for optimizing V
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
%------------------------ Output Variables ---------------------------------
% Output the following quantities for the largest steepness zeta
% depth           - data depth at convergence
% time            - computational time
% Vopt            - the projection direction of residuals 
%==========================================================================
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
%==========================================================================
debug = 0;
%==========================================================================
% We have implemented all the following methods
% Further explanations
% The main types of the algorithms include 
% (i) manifold optimization using the Manopt package (tr: trust region; cg: conjugate gradient; bb: Barzilai-Borwen; rlbfgs: Riemannian limited memory BFGS), 
% (ii) Riemannian geodesic gradient (0: without line search; 1: with line search), 
% (iii) Procrustes rotation (0: without line search; 1: with line search),
% (iv) accelerated Procrustes rotation (0: without line search; 1: with line search), 
% (v) second accelerated Procrustes rotation. 
%     0: no line search; 1: with line search
%     method1        Manopt:tr             trustregion with explicit hessian***
%     method2        Manopt:cg             conjugategradient**
%     method3        Manopt:bb             barzilaiborwein
%     method4        Manopt:rlbfgs         Riemannian limited memory BFGS**
%     method5        SAP:pr0            Procrustes rotation without linesearch
%     method6        SAP:pr1            Procrustes rotation with linesearch
%     method7        SAP:apr0           Accelerated procrustes rotation without linesearch
%     method8        SAP:apr1           Accelerated procrustes rotation with linesearch**
%     method9        SAP:2ndAcc         2nd_Accelerated procrustes rotation with linesearch    
%     method10       SAP:rgg0           Riemannian geodesic gradient withoutlinesearch
%     method11       SAP:rgg1           Riemannian geodesic gradient with linesearch*
[n, p] = size(X);
[~, m] = size(R);

if ~exist('ZeroOut', 'var')
    ZeroOut =[];
end
ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);
  
if  ~isfield(opts, 'scheme') || isempty(opts.scheme )
    opts.scheme  =  'Manopt:tr';  %'Manopt:cg', 'Manopt:rlbfgs', 'SAP:rgg1', 'SAP:pr1' , 'SAP:apr1', 'SAP:2ndAcc'
    if n <= 1000 && p * m < 500  
        opts.scheme  =  'Manopt:tr'; 
    elseif  n <= 1000 &&  p * m >= 500
        opts.scheme  =  'SAP:2ndAcc';
    else % n > 10000
        opts.scheme  =  'Manopt:rlbfgs';
    end
end

    
%==========================================================================
if isempty(zetaGrid)
    zetaGrid = 1.25.^linspace(0,10,11); 
    gridsize = length(zetaGrid);
else
    zetaGrid = unique(sort(zetaGrid, 'ascend')); 
    gridsize = length(zetaGrid);
    % warning('Remove redundant zetas in the grid.')
end

tic;
for i = 1:gridsize
    zeta = zetaGrid(i);
    if i == 1
        NumInits = opts.numOfInitValues; 
        Vres =  zeros(p, m, NumInits);
        fres = zeros(NumInits, 1);
        depth = zeros(NumInits, 1);
        [initVs] = Funct_InitVs(Xt, R, NumInits, ZeroOut);
        for j = 1:NumInits
            V_init = initVs(:, :, j); 
            [Vres(:,:,j), depth(j), fres(j)] = Func_PHD_InAnn(X, R, Xt, V_init, zeta, opts, ZeroOut);
        end
        optInd = find(depth == min(depth), 1); % depth value rather than the value of the current function 
%         optInd = find(fres == min(fres), 1);
       
        V = Vres(:, :, optInd); %Select the V that minimize the function value!
        depth = depth(optInd); 
    else
        V_init = V;  % Use warmstarting for V optimization
        [V, depth, ~, iternum] = Func_PHD_InAnn(X, R, Xt, V_init, zeta, opts, ZeroOut);
    end
    if debug > 0
        fprintf('Steepness zeta=%d finished \n', zeta)
    end
end
time = toc;
Vopt = V;
end


function [V, depth, final_cost, iternum] = Func_PHD_InAnn(X, R, Xt, V_init, zeta, opts, ZeroOut)
%==========================================================================
% This function computes the datadepth for a given zeta assuming the phi
% function is differentiable
if strcmpi(opts.scheme, 'SAP:Pr') || strcmpi(opts.scheme, 'SAP:Apr')  || strcmpi(opts.scheme, 'SAP:2ndAcc')
    refL = 4*zeta^2*(norm(X,'fro')*norm(R,'fro'))^2/(3*sqrt(3));
elseif strcmpi(opts.scheme, 'SAP:Rgg')
    refL = 4*zeta^2*(norm(X,'fro')*norm(R,'fro'))^2/(3*sqrt(3)) + zeta*norm(X,'fro')*norm(R,'fro');
else
    refL = 1;
end

% Call the main PHD algorithm on the differentiable loss
[V, iternum, final_cost] = Func_PHD(X, R, Xt, V_init, zeta, refL, opts, ZeroOut);
%==========================================================================
%The depth is computed according to the exact non-negative indicator funtion
% Xdiag =  sum((X*V).*R, 2);
% nonnegative_Xdiag = heaviside(Xdiag) + 1/2 * (Xdiag==0);
% depth = sum(nonnegative_Xdiag) / size(X,1);
depth = Funct_01DepthVal(X, R, V);
end




