%% This is a demo for computing GLM data depth
%
% The two main steps (after generaing the data) below are 
%   Funct_GLMResidual_Gradient (to form the R matrix), and Func_PHD01 (for
%   computing d01)

clear; close all; clc;
addpath(genpath(pwd)) % Make sure to include the direction into Matlab path
rng('default')
rng(0)


%% Specify problem dimensions and output settings
% distribution family
family = 'Gaussian'; % 'poisson'; %  'Gaussian'
% problem size
dimsetting = 'small';     %'small'; or 'med'; or 'med-Large'  
switch lower(dimsetting)
    case 'small'
        n = 500;
        p = 10;
    case 'med'
        n = 1000;
        p = 100;
    case 'med-large'
        n = 1000;
        p = 500;
end
% number of responses
m = 10; %10; % 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Simulation data parameters
NumRuns = 20; % number of simulations (runs)
Output = 0;
dataName = 'mynorm-highcorr'; 
BTrue = ones(p, m);
% parameters for outlier generation 
% addoutlier controls the percentage and outlier strength and leverage strength of outliers
addoutlier.Pct = 0;
addoutlier.outlierStrength = 0;
addoutlier.levStrength = 0;

% Specify a parameter value (B^0 in the paper, of size p x m) to calculate depth
TargetType = 'equipoint-0.8'; % 'true';  % 'random'; % 'equipoint-5'; % i.e., [5,5,...5]% % 'equipoint-0.1'; % i.e., [0.1,0.1,...0.1]%
if strcmpi(TargetType, 'true')
    B_target = BTrue;
elseif strcmpi(TargetType, 'random')
    rng(0)
    B_target = randn(p,m);
elseif strcmpi(TargetType(1:9), 'equipoint')
    B_target = ones(p,m) * str2num(TargetType(11:end)); 
else
    error('Specify B')
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Specify an optimization method and parameters  
methodList = {'SAP:2ndAcc'}; % methodList = {'SAP:2ndAcc'};   % 
    % The methods we have implemented: {'Manopt:tr', 'Manopt:cg', 'Manopt:bb', 'Manopt:rlbfgs', 'SAP:pr0', 'SAP:pr1', 'SAP:apr0', 'SAP:apr1', 'SAP:2ndAcc', 'SAP:rgg0', 'SAP:rgg1'};

tol = 1e-3;
tolgrad = 1e+0;
maxiter = 5e+3;

problemsize = [n, p, m];
numOfInitValues = 10;   % This parameter controls the number of (random) initial values. 
                        % Multiple random starts, together with our
                        % optimization techniques,  are critical for tackling the nonconvexity.
                        % In some very challenging problems, one can consider using a larger numOfInitValues

OPTS=cell(1,length(methodList));
for j = 1:length(methodList)
        method = methodList{j};
        optsVOpt = Funct_ParamsConfiguration(problemsize, tol, tolgrad, maxiter, numOfInitValues, [], method);
        OPTS{j}=optsVOpt;     
end
zetaGrid = []; zetaGrid = []; % the annealing grid of zeta's (which is necessary for calculating Tukey's type depth). Use the default 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Start the computation
fprintf('NumRuns = %d, n = %d, m = %d, p = %d, design = %s, family = %s. \n',  NumRuns, n, m, p, dataName, family);
% fprintf('gridsize for steepness zeta = %d. \n', numel(zetaGrid));
fprintf('tolgrad = %.e, maxiter = %.e, numOfInitValues = %d \n', tolgrad, maxiter, numOfInitValues);
fprintf('================================================================================ \n');
fprintf('================================================================================ \n');

depthres = zeros(NumRuns, length(methodList)); % to compare different methods for comparisons in total
timeres = zeros(NumRuns, length(methodList));
iternumes = zeros(NumRuns, length(methodList));
% Generate multiple datasets, call each method, and summarize the results
for i = 1:NumRuns
    %% Generate some data
    fprintf('================================================================================ \n');
    fprintf('%dth simulation dataset  ... \n', i);
    [X, Y] = Funct_geneGLMData(dataName, n, p, BTrue, addoutlier, family);
    Xt = X.'; 
    %--------------------------------------------------------------------------------%
    % Construct the R matrix
    R = Funct_GLMResidual_Gradient(Y, X, B_target, family); 
        % This is the only quantity that relies in the distribution or specified loss
    %--------------------------------------------------------------------------------%    
    for j = 1:length(methodList)
        [depth, time, Vopt, iternum] = Func_PHD01(X, R, Xt, zetaGrid, OPTS{j}); % Main function
        fprintf('The depth the using %s method is %.2f. Time elapsed %.3f. Iteration number is %d.\n', methodList{j}, depth, time, iternum)
        depthres(i,j) = depth;
        timeres(i,j) = time;
        iternumes(i,j) = iternum;
    end
end

for j = 1:length(methodList)
    fprintf('================================================================================ \n');
    % fprintf('The results using %s method are: \n', methodList{j})
    fprintf('The mean and median depths are %.3f and %.3f. \n', mean(depthres(:,j)), median(depthres(:,j)));
    fprintf('The mean and median computational time are %.2f and %.2f. \n', mean(timeres(:,j)), median(timeres(:,j)));
    fprintf('The mean and median iteration number are %.2f and %.2f. \n', mean(iternumes(:,j)), median(iternumes(:,j)));
end
