function [V, final_cost, info, iternum] = Func_ManOpt(X, Xt, V_init, R, zeta, opts, ZeroOut)
%==========================================================================
% This function uses some Manifold optimization methods to solve the
% polished data depth problem.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% initV           - initial value of V
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% opts            - options for V optimization
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
%------------------------ Output Variables ---------------------------------
% V               - optimal solution of V
% final_cost      - function value at convergence
% info 	          - progress information
%==========================================================================

[p,m] = size(V_init);
% Create the problem structure.
manifold = spherefactory(p,m);
problem.M = manifold;

if ~exist('ZeroOut', 'var')
    ZeroOut =[];
end
ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);
if ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained')
    error('Not implemented yet; use the penalized method')
end

% Define the problem cost function and its Euclidean gradient.
problem.cost  = @(V) cost(X, Xt, V, R, zeta, ZeroOut);

problem.egrad = @(V) egrad(X, Xt, V, R, zeta, ZeroOut);

if isfield(opts, 'usehess') && strcmpi(opts.usehess, 'TRUE')
    problem.ehess = @(V,U) ehess(X, Xt, V, R, zeta, ZeroOut, U);
end

if isfield(opts, 'checkgrad') && strcmpi(opts.checkgrad, 'TRUE')
    % Numerically check gradient consistency (optional).
    fprintf('=====================================================\n')
    fprintf('Checking Gradient.\n')
    checkgradient(problem);
end

if isfield(opts, 'usehess') && isfield(opts, 'checkhess') && strcmpi(opts.checkhess, 'TRUE')
    fprintf('=====================================================\n')
    fprintf('Checking Hessian.\n')
    checkhessian(problem);
end

% Solve the problem
switch opts.ManoptMethod
    case 'trustregion'
        [V, final_cost, info, ~, iternum] = trustregions(problem, V_init, opts);
    case 'conjugategradient'
        [V, final_cost, info, ~, iternum] = conjugategradient(problem, V_init, opts);
    case 'barzilaiborwein'
        [V, final_cost, info, ~, iternum] = barzilaiborwein(problem, V_init, opts);
    case 'rlbfgs'
        [V, final_cost, info, ~, iternum] = rlbfgs(problem, V_init, opts);
    otherwise
        error('Please specify the method for manifold optimization.')
end


if ZeroOutEffective       % add a calibration to ensure certain position to be 0
    if max(max(abs((V .* (~ZeroOut.mat))))) > 1e-2
        warning('The penalty did not work')
    end
    V = V .* ZeroOut.mat;
    V = V / (norm(V, 'fro')); 
end

end

%% Define the objective function value, euclidean grad and euclidean hessian
function [f] = cost(X, Xt, V, R, zeta, ZeroOut)
[f] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta, ZeroOut);
end

function [g] = egrad(X, Xt, V, R, zeta, ZeroOut)
[~,g] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta, ZeroOut);
end

function [h] = ehess(X,Xt,V,R,zeta,ZeroOut, U)
[~,~,h] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta, ZeroOut, U);
end


