function[Vopt, iternum, final_cost]= Func_PHD(X, R, Xt, V_init, zeta, L, opts, ZeroOut)
%==========================================================================
% This function computes the optimal V for polished data depth using some 
% methods for smooth optimization. 
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% R               - residual matrix
% Xt              - transpose of X
% initV           - initial value of V
% zeta            - 'inverse-cooling' parameter for annealing, which controls
%                   the steepness of the function in approximating the sign function 
% L               - The given lipschitz constant. 
% opts            - Options for optimizing V
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
%------------------------ Output Variables ---------------------------------
% Output the following quantities for the largest steepness zeta
% depth           - data depth at convergence
% time            - computational time
% Vopt            - the projection direction of residuals 
%==========================================================================
%==========================================================================

if ~exist('ZeroOut', 'var')
    ZeroOut =[];
end
ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);

if ~isfield(opts, 'maxiter')
    opts.maxiter = 1000;
end
if ~isfield(opts, 'tol')
    opts.tol = 1e-8;
end
if ~isfield(opts, 'tolgrad')
    opts.tolgrad = 1e-6;
end
if ~isfield(opts, 'beta')
    opts.beta = 2;
end
if isempty(L) 
   if strcmpi(opts.linesearch, 'TRUE')       
       if strcmpi(opts.scheme, 'SAP:Pr')
           L = 1e+13;
       elseif strcmpi(opts.scheme, 'SAP:Apr')
           L = 1e+14;
       elseif strcmpi(opts.scheme, 'SAP:2ndAcc')
           L = 1e+6;
       elseif strcmpi(opts.scheme, 'SAP:Rgg')
           L = 1e+8;
       end
   else
       error('Please provide the value of L')
   end
end

if ~strcmpi(opts.scheme, 'ManOpt')    
   if ~isfield(opts, 'linesearch') || ~strcmpi(opts.linesearch,'TRUE')
       opts.linesearch = 'FALSE';
   end
   if ~isfield(opts, 'alpha')
       opts.alpha = 1e-2;
   end
end

if strcmpi(opts.scheme, 'ManOpt')
    [Vopt, final_cost, ~, iternum] = Func_ManOpt(X, Xt, V_init, R, zeta, opts, ZeroOut);
elseif strcmpi(opts.scheme, 'SAP:Apr')
    [Vopt, final_cost, ~, iternum] = Func_APG(X, Xt, V_init, R, zeta, L, opts, ZeroOut);
elseif strcmpi(opts.scheme, 'SAP:Pr')
    [Vopt, final_cost, ~, iternum] = Func_PG(X, Xt, V_init, R, zeta, L, opts);
elseif strcmpi(opts.scheme, 'SAP:2ndAcc')
    [Vopt, final_cost, ~, iternum] = Func_2ndAcc(X, Xt, V_init, R, zeta, L, opts, ZeroOut);
elseif strcmpi(opts.scheme, 'SAP:Rgg')  
    [Vopt, final_cost, ~, iternum] = Func_RGG(X, Xt, V_init, R, zeta, L, opts);    
end

end