function [V, final_cost, info, i] = Func_RGG(X, Xt, V_init, R, zeta, L, opts)
%==========================================================================
% This function performs Riemannian geodesic gradient with backtracking line search
% to solve the polished data depth problem.
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% initV           - initial value of V
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% L               - lipschitz constant
% opts            - options for V optimization
%------------------------ Output Variables ---------------------------------
% V               - optimal solution of V
% final_cost      - function value at convergence
% info 	          - progress information
%==========================================================================
debug = 1;
f = zeros(1,opts.maxiter);
g = zeros(1,opts.maxiter);  %gradnorm
V = V_init;
alpha = opts.alpha;
beta = opts.beta;



for i = 1:opts.maxiter
    [f_cur, g_eu_cur] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta); %Euclidean gradient
    g_cur = g_eu_cur - sum(sum(g_eu_cur.*V, 2))*V; % the riemannian gradient
    % g_cur - trace(V.'*g_cur)*V;
    g_cur_norm = norm(g_cur,'fro');
    
    if strcmpi(opts.linesearch,'TRUE')
        rho = max(alpha*L, g_cur_norm/pi);
        while true
            U_norm = g_cur_norm/rho;
            V_new =  cos(U_norm)*V - (sin(U_norm)/g_cur_norm)*g_cur;
            [sur] = surrogate(f_cur, g_cur, rho);
            [f_new] = Funct_ObjGradHessian_V(X, Xt, V_new, R, zeta);
            if f_new < sur
                V = V_new;
                break
            else
                if rho > L
                    V = V_new;
                    break
                end
            end
            rho = rho*beta;            
        end
    else
        U_norm = g_cur_norm/L;
        V = cos(U_norm)*V - (sin(U_norm)/g_cur_norm)*g_cur;
    end
    f(i) = f_cur;
    g(i) = g_cur_norm;
    if i > 1 && g(i) < opts.tolgrad
        f = f(1:i);
        g = g(1:i);
        break
    end
    if i > 1 && abs(f(i)-f(i-1))<opts.tol
        f = f(1:i);
        g = g(1:i);
        break
    end    
    if debug > 0 && i > 1 && f_cur > f(i-1) 
        warning('Function value is not decreasing. Try a smaller step-size.')
    end
    

end
final_cost = f(end);
info.cost = f;
info.gradnorm = g;
end


%% Define the surrogate function
function [sur] = surrogate(f_cur, g_cur, rho)
sur = f_cur - 1/(2*rho)*norm(g_cur,'fro')^2;
end
