 function [Best, alphaest, Sest, Vest, ks] = Func_RRR(X, Y, ks, estimateIntercept)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Reduced Rank Regression (RRR) 
% An RRR is fit by regressing Y (n x m) on X (n x p), where the coefficient 
% matrix B is subject to a low-rank constraint: r(B) <= ks. 
% The intercept is not subjected to any regularization.
%
% Notes: 1. The input X must not include the intercept column.
%        2. The corresponding coefficient matrix B will not contain 
%           intercept estimate. We will provide a separate intercept
%           estimate alpha as a column vector.
%        3. That is, the fitted model is Y = 1 * alpha' + X * B.
%
% Input arguments:
%   X: The design matrix (no intercept column)
%   Y: The response matrix
%   ks: The rank value(s) of interest (can contain more than one rank values)
%   estimateIntercept: (Default: 1)
%       0: No intercept estimation (return zeros).
%       1: The intercept alpha will be simultaneously estimated.      
%
% Output arguments:
%   Best: The optimal B estimate with rank less than ks (or the
%         solution path of B's if numel(ks) > 1).
%   alphaest: The intercept estimate vector of size m x 1 (or the solution
%             path of multiple estimates [augmented at size m x numel(ks)]).
%   Sest: The loading matrix of size p x r (or the solution path of S's, in
%         a cell array if numel(ks) > 1). 
%   Vest: the rotation matrix of size m x r satisfying V'V = I (or the 
%         solution path of V's, in a cell array if numel(ks) > 1).
%   ks: the true rank upper bound(s) when solving the RRR optimization 
%       (which might be smaller than the input ks).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% 1. Set default value for intercept estimation
if ~exist('estimateIntercept', 'var') || isempty(estimateIntercept)
    estimateIntercept = 1;
end

% 2. ---------Preprocessing steps----------------
m = size(Y, 2); p = size(X, 2);
rmax = min([m, rank(X,1e-5)]); 
% r(X) is used because r(B_unpen)<=r(X)

% Truncate the ks. Larger values result in the same full rank estimates
ks(ks >= rmax) = rmax; 

% X_ori = X; Y_ori = Y; % might be used later. (Not required currently.)
% Center X and Y to estimate the coefficient matrix when estimateIntercept == 1 
if estimateIntercept == 1
    Y_colmeans = mean(Y);
    Y = bsxfun(@minus, Y, Y_colmeans);
    X_colmeans = mean(X);
    X = bsxfun(@minus, X, X_colmeans);
end

% Precalculation for obtaining RRR estimate/solution path
tmpUniEst = X' * Y;
tmpXGInv = pinv(X); % We will not use '\'
tmpSigmaGInv = tmpXGInv * tmpXGInv';
B_unpen = tmpSigmaGInv * tmpUniEst; % the OLS estimate

eigTarget = tmpUniEst' * B_unpen;
[eigV0, eigD0] = eig((eigTarget+eigTarget')/2);
[~, IX] = sort(diag(eigD0), 'descend');
eigV = eigV0(:, IX);

% 3.-----------------RRR algorithm------------------
numEsts = numel(ks);
if numEsts == 1 % if ks is singleton
    k = ks;
    Vest = eigV(:, 1:k);
    Sest = B_unpen * Vest;
    Best = Sest * Vest'; 
    if estimateIntercept == 1
        alphaest = Y_colmeans - X_colmeans * Best;
    	alphaest = alphaest';
    else
        alphaest = zeros(m,1);
    end
else % if ks has multiple values of ranks, we provide the complete solution path
    Best = zeros(p, m, numEsts);
    alphaest = zeros(m, numEsts); % 2D array
    Sest = cell(1, numEsts); 
    Vest = cell(1, numEsts);
    for kInd = 1:numel(ks)
        k = ks(kInd);
        Vest{1, kInd} = eigV(:, 1:k);
        Sest{1, kInd} =  B_unpen * Vest{1, kInd};
        Best(:, :, kInd) = Sest{1, kInd} * Vest{1, kInd}';
        if estimateIntercept == 1
            alphaestt = Y_colmeans - X_colmeans * Best(:, :, kInd); 
            % need separate variable as alphaest is already defined
            alphaest(:, kInd) = alphaestt';
        end
    end
end    
end

