 function [depth, time, Vopt, L_unscaled, L_scaled, iternum] = Func_RRR_Depth(X, Y, B, R, r, zetaGrid, opts, rho_depth)
%==========================================================================
% This function computes the RRR data depth using the successive
% optimization scheme (annealing). The implementation of the slacked data depth
% follows the same lines as Func_Sparsity_Depth. 
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Y               - response matrix
% Xt              - transpose of X
% B               - target estimate
% R               - residual matrix
% r               - the rank of the RRR depth, also the rank of B
% zetaGrid        - the grid of the steepness for successive apporximations of the sign function
% opts            - Options for optimizing V and L
% rho_depth       - a scale parameter aded in the objective function
%                   (Theoretically it should show no difference in RRR depth, but
%                   practically, due to our approximation scheme, a good
%                   empirical choice can lead to smaller objective function
%                   values; see the note below.)
%------------------------ Output Variables ---------------------------------
% Output the following quantities for the largest steepness zeta
% depth           - data depth at convergence
% time            - computational time
% Vopt            - the optimal projection direction of the residuals/influences 
% Lopt            - the optimal (repametrized) slack variables/parameters 
%==========================================================================
debug = 0;
%==========================================================================

[n, p] = size(X);
[~, m] = size(R);
Xt = X';

[P, D, Q] = svd(B);

P_perp = P(:, r+1:end);
L_init = D(r+1:end, r+1:end); 
Q_perp = Q(:, r+1:end);  
% lambda = D(r, r);


[p_L, m_L] = size(L_init);

if ~exist('rho_depth', 'var') ||  isempty(rho_depth)
    % Make a default choice of rho
    rho_depth = 1e-4 * norm(X, 2)^2; %0.025; %0.05;
    % Note: Theoretically, the 0-1 depth should be invariant to rho_depth, but
    % because we use a progressive approximation scheme, rho_depth shows
    % some numerical differences. Experience shows that this default choice
    % often leads to a small depth during the optimization (if not the
    % smallest)
end

if isempty(opts)
    opts.V_method = '2ndAcc';
    opts.L_method = '2ndAcc';

    opts.numOfInitValues = 10;
    opts.maxiter = 3e+3;
    opts.tolgrad = 1e-5;
    opts.tol = 1e-3;
    opts.V_tol = 1e-4;
    opts.L_tol = 1e-4;
    opts.V_bcd_maxiter = 100;
    opts.L_bcd_maxiter = 100;
    
end

if isempty(zetaGrid)
    zetaGrid = 1.25.^linspace(0,10,11); 
    gridsize = length(zetaGrid);
else
    zetaGrid = unique(sort(zetaGrid, 'ascend')); 
    gridsize = length(zetaGrid);
    % warning('Remove redundant zetas in the grid.')
end

% Construct lambda, which depends on the given B
lambda = norm((1 / rho_depth) * P_perp' * X' * (X * B - Y) * Q_perp, 2);

if debug > 0 
    fprintf('The value of rho_depth is %.4f. \n', rho_depth); 
end

tic;
for i = 1:gridsize
    zeta = zetaGrid(i);
    if i == 1
        NumInits = opts.numOfInitValues; 
        Vres =  zeros(p, m, NumInits);
        Lres = zeros(p_L, m_L, NumInits);
        n_iter = zeros(NumInits, 1);
        depth = zeros(NumInits, 1);
        [initVs] = Funct_InitVs(Xt, R, NumInits);
        for j = 1:NumInits
            V_init = initVs(:, :, j); 
            L_init = L_init;
            [Vres(:,:,j), Lres(:,:,j), depth(j), n_iter(j)] = Func_RRR_Depth_InAnn(X, Xt, V_init, R, P_perp, L_init, Q_perp, lambda, opts, zeta, rho_depth);
        end
        optInd = find(depth == min(depth), 1);
       
        V = Vres(:, :, optInd); %Select the V that minimize the function value!
        L = Lres(:, :, optInd);
        depth = depth(optInd);
    else
        V_init = V;  % Use warmstarting for V optimization
        L_init = L;
        [V, L, depth, iternum] = Func_RRR_Depth_InAnn(X, Xt, V_init, R, P_perp, L_init, Q_perp, lambda, opts, zeta, rho_depth);
    end
    if debug == 1
        fprintf('Steepness zeta=%d finished \n', zeta)
    end
end
time = toc;
Vopt = V;
% Lopt = L;
L_scaled = L;
L_unscaled = L * rho_depth;

% [~, ~, depth] = Funct_Obj_RRR_Depth(X, Vopt, R, P_perp, Lopt, Q_perp, zeta, rho_depth);

end


function [V, L, depth, iternum] = Func_RRR_Depth_InAnn(X, Xt, V_init, R, P_perp, L_init, Q_perp, lambda, opts, zeta, rho_depth)
% Use BCD to solve the (V, L) optimization problem.
% (The loss constructed from aux_Phi is differentiable)
% loss 
[V, L, iternum] = Func_RRR_Depth_InAnn_BCD(X, Xt, V_init, R, P_perp, L_init, Q_perp, lambda, opts, zeta, rho_depth);
%==========================================================================
% Now return the 0-1 depth (using 1_{>=0} as the measure) 
[~, depth] = Funct_Obj_RRR_Depth(X, V, R, P_perp, L, Q_perp, zeta, rho_depth, 1);
end




